<?php
/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License version 3.0
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/AFL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License version 3.0
 */
if (!defined('_PS_VERSION_')) {
    exit;
}

/**
 * @method public void        __construct()
 * @method public array       getOrder(int $orderId)
 * @method public array       getOrderIds(string $syncStatusesString, int $page, int $size, string $updatedAfter)
 * @method public array       mapOrderProducts(array $orderProducts)
 * @method public array       mapOrder(array $data)
 */
class WingOperationsOrders
{
    protected $context;

    /* Class methods */
    /**
     * @method __construct
     *
     * Constructor
     *
     * @return void
     */
    public function __construct()
    {
        $this->setContext();
    }

    /**
     * @method public uninstall
     *
     * @return Context
     */
    private function setContext()
    {
        $this->context = Context::getContext();
    }

    /**
     * @method public getOrder
     *
     * @param int $orderId
     *
     * @return array
     */
    public function getOrder($orderId)
    {
        $db = Db::getInstance();

        /* Get Order information */
        $order = new Order($orderId);

        /* Get Order Delivery Address */
        $deliveryAddressId = (int) $order->id_address_delivery;
        $deliveryAddress = new Address((int) $deliveryAddressId);

        /* Get Order Customer */
        $customer = new Customer((int) $order->id_customer);

        /* Get Carrier */
        $carrierId = $order->id_carrier;
        $carrier = new Carrier((int) $carrierId);

        /* Get Products from Order */
        $products = [];
        if (version_compare(_PS_VERSION_, '1.7.4', '>')) {
            // Récupérer les détails des produits dans la commande
            $products = $order->getOrderDetailList();
            $currency = new Currency($order->id_currency);

            // Récupérer les règles de panier (codes promo) appliquées
            $cartRules = $order->getCartRules();
            $totalCartRuleDiscount = 0;

            foreach ($cartRules as $rule) {
                $totalCartRuleDiscount += $rule['value_tax_excl']; // Total des remises appliquées par les codes promo
            }

            if ($totalCartRuleDiscount > 0) {        
                // Calculer le prix total HT des produits pour répartir proportionnellement
                $totalProductsPriceHT = 0;
        
                foreach ($products as $detail) {
                    $totalProductsPriceHT += $detail['total_price_tax_excl'];
                }
        
                // Répartition des remises HT par produit
                foreach ($products as &$detail) {
                    $productTotalHT = $detail['total_price_tax_excl'];
                    $quantity = $detail['product_quantity'];

                    // Calcul de la remise totale pour ce produit
                    $productDiscountHT = ($productTotalHT / $totalProductsPriceHT) * $totalCartRuleDiscount;

                    // Calcul de la remise unitaire
                    $unitDiscountedPriceHT = round($productDiscountHT / $quantity, 2);
                    $finalProductPriceHT = $detail['unit_price_tax_excl'] - $unitDiscountedPriceHT;

                    $detail['unit_price_tax_excl'] = round($finalProductPriceHT, 2);
                }
            }
        } else {
            $products = $order->getProducts();
            if (0 == count($products)) {
                $products = $order->getCartProducts();
            }
        }

        /* Map entities to the final Order object */
        $obj = new stdClass();
        $obj->order = $order;
        $obj->deliveryAddress = $deliveryAddress;
        $obj->customer = $customer;
        $obj->carrier = $carrier;
        $obj->products = $products;

        return $obj;
    }

    /**
     * @method public getOrderIds
     *
     * @param string $syncStatutesString
     * @param string $updatedAfter
     * @param int $shopId
     *
     * @return array
     */
    public function getOrderIds(
        $syncStatuses,
        $updatedAfter,
        $shopId,
        $page,
        $size
    ) {
        $db = Db::getInstance();
        $syncStatusString = ' AND `current_state` IN (' . implode(', ', array_map('intval', explode(',', $syncStatuses))) . ')';
        $updatedAfterString = ' AND `date_upd` >= \'' . date($updatedAfter) . '\'';

        $offset = ($page - 1) * $size;
        $getOrderIdsRequest = new DbQuery();
        $orderIds = $db->executeS(
            $getOrderIdsRequest
                ->select('id_order')
                ->from('orders')
                ->where(
                    'id_shop = ' . (int) $shopId .
                    $syncStatusString .
                    $updatedAfterString
                )
                ->limit((int) $size, (int) $offset)
        );

        $getOrderIdsTotalCountRequest = new DbQuery();
        $totalOrderIds = count(
            $db->executeS(
                $getOrderIdsTotalCountRequest
                    ->select('id_order, id_cart, id_carrier')
                    ->from('orders')
                    ->where(
                        'id_shop = ' . (int) $shopId .
                        $syncStatusString .
                        $updatedAfterString
                    )
            )
        );

        $orderIdsValues = [];
        foreach ($orderIds as $order) {
            $orderIdsValues[] = $order['id_order'];
        }

        return [
            'count' => $totalOrderIds,
            'shopId' => $shopId,
            'page' => $page,
            'size' => $size,
            'orderIds' => $orderIdsValues,
        ];
    }

    /**
     * @method private mapOrderProducts
     *
     * @param array $orderProducts
     *
     * @return array
     */
    private function mapOrderProducts($orderProducts)
    {
        $products = [];
        foreach ($orderProducts as $orderProductToMap) {
            $orderProduct = (object) $orderProductToMap;
            $products[] = [
                'total_price_tax_incl' => (float) $orderProduct->total_price_tax_incl,
                'total_price_tax_excl' => (float) $orderProduct->total_price_tax_excl,
                'unit_price_tax_incl' => (float) $orderProduct->unit_price_tax_incl,
                'unit_price_tax_excl' => (float) $orderProduct->unit_price_tax_excl,
                'product_name' => $orderProduct->product_name,
                'product_id' => (int) $orderProduct->product_id,
                'product_ean13' => property_exists($orderProduct, 'product_ean13') ? $orderProduct->product_ean13 : '',
                'product_upc' => property_exists($orderProduct, 'product_upc') ? $orderProduct->product_upc : 0,
                'product_quantity' => property_exists($orderProduct, 'product_quantity') ? (int) $orderProduct->product_quantity : 0,
                'product_price' => property_exists($orderProduct, 'product_price') ? (float) $orderProduct->product_price : 0,
                'product_reference' => property_exists($orderProduct, 'product_reference') ? $orderProduct->product_reference : '',
                'width' => property_exists($orderProduct, 'width') ? (float) $orderProduct->width : 0,
                'height' => property_exists($orderProduct, 'height') ? (float) $orderProduct->height : 0,
                'depth' => property_exists($orderProduct, 'depth') ? (float) $orderProduct->depth : 0,
                'weight' => property_exists($orderProduct, 'product_weight') ? (float) $orderProduct->product_weight : 0,
            ];
        }

        return $products;
    }

    /**
     * @method private mapPartner
     *
     * @param int $orderId
     *
     * @return string
     */
    private function mapPartner($orderId)
    {
        $partner = null;

        $db = Db::getInstance();
        if (Module::isInstalled('shoppingfeed')) {
            $getZalandoSANSPartnerRequest = new DbQuery();
            $marketplaceName = $db->executeS(
                $getZalandoSANSPartnerRequest
                    ->select('name_marketplace')
                    ->from('shoppingfeed_order')
                    ->where('id_order = ' . (int) $orderId)
                    ->limit(1, 0)
            );

            $partner = strpos($marketplaceName[0]['name_marketplace'], 'zalando') ? 'ZALANDO_STORE_AND_SUPPLY' : null;
        }

        if (Module::isInstalled('crossmods')) {
            $getZalandoAETLPartnerRequest = new DbQuery();
            $marketplaceName = $db->executeS(
                $getZalandoAETLPartnerRequest
                    ->select('marketplace')
                    ->from('ps_xm_orders')
                    ->where('id_order_ps = ' . (int) $orderId)
                    ->limit(1, 0)
            );

            $partner = strpos($marketplaceName[0]['marketplace'], 'Zalando') ? 'ZALANDO' : null;
        }

        return $partner;
    }

    /**
     * @method private mapUJAShipFromStore
     *
     * @param int $orderId
     *
     * @return string
     */
    private function mapUJAShipFromStore($orderId)
    {
        $db = Db::getInstance();
        $getUJAStoreRequest = new DbQuery();
        $response = $db->executeS(
            $getUJAStoreRequest
                ->select('fastmag')
                ->from('uja_ship_from_store')
                ->where('id_order = ' . $orderId)
        );

        $storeName = $response[0]['fastmag'];

        return $storeName;
    }

    /**
     * @method private mapAutoAssignToCollect
     *
     * @param int $orderId
     *
     * @return string
     */
    private function mapAutoAssignToCollect($orderId, $shopId)
    {
        $storeName = null;
        $autoAddToCollect = (bool) Configuration::get(
            'WING_SHOP_' . $shopId . '_AUTO_ADD_TO_COLLECT'
        );
        $wingPickupName = Configuration::get(
            'WING_SHOP_' . $shopId . '_PICKUP_NAME'
        );

        if ($wingPickupName) {
            $storeName = $wingPickupName;
        }

        if (Module::isInstalled('uja_ship_from_store') && $autoAddToCollect == 1) {
            $storeName = $this->mapUJAShipFromStore($orderId);
        }

        return $storeName;
    }

    /**
     * @method private mapOrderWingCarrier
     *
     * @param int $carrierId
     *
     * @return string
     */
    private function mapOrderWingCarrier($carrierId, $orderCartId, $orderId)
    {
        /* Define context vars */
        $shopId = (int) $this->context->shop->id;
        $psCarrier = new Carrier($carrierId);

        /* Default Wing carrier settings */
        $pickupId = null;
        $email = null;

        if ($psCarrier->external_module_name == 'colissimo' && Module::isInstalled('colissimo')) {
            if (null == $pickupId) {
                $pickupId = Db::getInstance()->getValue(
                    'SELECT pickup.colissimo_id FROM ' .
                        _DB_PREFIX_ .
                        'colissimo_pickup_point pickup LEFT JOIN ' .
                        _DB_PREFIX_ .
                        'colissimo_cart_pickup_point cartpickup ON ' .
                        ' (cartpickup.id_colissimo_pickup_point = pickup.id_colissimo_pickup_point)' .
                        ' WHERE cartpickup.id_cart = ' .
                        (int) $orderCartId
                );
            }
        }

        if ($psCarrier->external_module_name == 'socolissimo' && Module::isInstalled('socolissimo')) {
            $pickupId = Db::getInstance()->getValue(
                'SELECT `prid` FROM `' .
                    _DB_PREFIX_ .
                    'socolissimo_delivery_info` WHERE `id_cart` = ' .
                    (int) $orderCartId
            );
            $email = Db::getInstance()->getValue(
                'SELECT `ceemail` FROM `' .
                    _DB_PREFIX_ .
                    'socolissimo_delivery_info` WHERE `id_cart` = ' .
                    (int) $orderCartId
            );
        }

        if ($psCarrier->external_module_name == 'soflexibilite' && Module::isInstalled('soflexibilite')) {
            /* fallback * */
            if (null == $pickupId) {
                $pickupId = Db::getInstance()->getValue(
                    'SELECT `prid` FROM `' .
                        _DB_PREFIX_ .
                        'colissimo_delivery_info` WHERE `id_cart` = ' .
                        (int) $orderCartId
                );
                $email = Db::getInstance()->getValue(
                    'SELECT `ceemail` FROM `' .
                        _DB_PREFIX_ .
                        'colissimo_delivery_info` WHERE `id_cart` = ' .
                        (int) $orderCartId
                );
            }
        }

        if ('chronopost' == $psCarrier->external_module_name && Module::isInstalled('chronopost')) {
            $pickupId = Db::getInstance()->getValue(
                'SELECT `id_pr` FROM `' .
                    _DB_PREFIX_ .
                    'chrono_cart_relais` WHERE `id_cart` = ' .
                    (int) $orderCartId
            );
        }

        if ('mondialrelay' == $psCarrier->external_module_name && Module::isInstalled('mondialrelay')) {
            $pickupId = Db::getInstance()->getValue(
                'SELECT `selected_relay_num` FROM `' .
                    _DB_PREFIX_ .
                    'mondialrelay_selected_relay` WHERE `id_cart` = ' .
                    (int) $orderCartId
            );
        }

        if ('tntofficiel' == $psCarrier->external_module_name && Module::isInstalled('tntofficiel')) {
            $tntRelay = Db::getInstance()->getValue(
                'SELECT `delivery_point` FROM `' .
                    _DB_PREFIX_ .
                    'tntofficiel_order` WHERE `id_order` = ' .
                    (int) $orderId
            );

            $withoutSlash = str_replace('\\', '', $tntRelay);
            $withoutIntro = str_replace('a:11:', '', $withoutSlash);

            preg_match('/(?<=s:5:")(\d{5}?)/', $withoutIntro, $matches);
            $pickupId = $matches ? $matches[0] : false;
        }

        /* Map entities to the final Order object */
        $obj = new stdClass();
        $obj->carrier = $psCarrier->name;
        $obj->pickupId = $pickupId;
        $obj->email = $email;

        return $obj;
    }

    /**
     * @method private mapOrder
     *
     * @param array $data
     *
     * @return array
     */
    public function mapOrder($data, $shopId)
    {
        $pickupName = $this->mapAutoAssignToCollect($data->order->id, $shopId);
        $wingCarrierInfo = $this->mapOrderWingCarrier(
            (int) $data->order->id_carrier,
            (int) $data->order->id_cart,
            (int) $data->order->id
        );
        $autoAddToCollect = (bool) Configuration::get(
            'WING_SHOP_' . $shopId . '_AUTO_ADD_TO_COLLECT'
        );

        return [
            'reference' => $autoAddToCollect == 1 ? $data->order->reference . '-' . $data->order->id : $data->order->reference,
            'id_order' => (string) $data->order->id,
            'id_shop_group' => (int) $data->order->id_shop_group,
            'id_shop' => (int) $data->order->id_shop,
            'current_state' => (int) $data->order->current_state,
            'carrier' => $wingCarrierInfo->carrier,
            'pickupId' => $wingCarrierInfo->pickupId,
            'autoCreateCollectFromPickupName' => $pickupName,
            'partner' => $this->mapPartner($data->order->id),
            'gift' => (bool) $data->order->gift,
            'gift_message' => $data->order->gift_message,
            'total_paid' => (float) $data->order->total_paid,
            'total_paid_tax_incl' => (float) $data->order->total_paid_tax_incl,
            'total_paid_tax_excl' => (float) $data->order->total_paid_tax_excl,
            'total_shipping' => (float) $data->order->total_shipping,
            'total_shipping_tax_incl' => (float) $data->order->total_shipping_tax_incl,
            'total_shipping_tax_excl' => (float) $data->order->total_shipping_tax_excl,
            'date_add' => $data->order->date_add,
            'currency' => 'EUR',
            'products' => $this->mapOrderProducts($data->products),
            'customer' => [
                'country' => Tools::strtoupper(
                    Country::getIsoById($data->deliveryAddress->id_country)
                ),
                'company' => $data->deliveryAddress->company,
                'lastname' => $data->deliveryAddress->lastname,
                'firstname' => $data->deliveryAddress->firstname,
                'address1' => $data->deliveryAddress->address1,
                'address2' => $data->deliveryAddress->address2,
                'postcode' => $data->deliveryAddress->postcode,
                'email' => $wingCarrierInfo->email
                    ? $wingCarrierInfo->email
                    : $data->customer->email,
                'city' => $data->deliveryAddress->city,
                'other' => $data->deliveryAddress->other,
                'phone' => $data->deliveryAddress->phone,
                'phone_mobile' => $data->deliveryAddress->phone_mobile,
                'vat_number' => $data->deliveryAddress->vat_number,
            ],
        ];
    }
}
