<?php
/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License version 3.0
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/AFL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License version 3.0
 */

/*
 *
 * @method public $this    setAuthToken(string $authToken)
 * @method public $this    setHTTPMethod(string $method)
 * @method public $this    setBaseURL(string $mode)
 * @method public $this    setPayload(array $payload)
 * @method public $this    setPayloadFormat(string $format)
 * @method public array    executeRequest()
 * @method private array   buildHeaders()
 * @method private string  buildURL()
 * @method private string  formatPayload()
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

require_once _PS_MODULE_DIR_ .
    DIRECTORY_SEPARATOR .
    'wing' .
    DIRECTORY_SEPARATOR .
    'classes' .
    DIRECTORY_SEPARATOR .
    'helpers.php';

class WingApiRequest
{
    /* Class constants */
    const POST = 'POST';
    const PUT = 'PUT';
    const DELETE = 'DELETE';
    const GET = 'GET';
    const PAYLOAD_FORMAT = 'json';
    const URLENCODED_FORMAT = 'urlencoded';

    /* Class parameters */
    private $baseURL = null;
    private $payload = [];
    private $HTTPMethod = WingApiRequest::GET;
    private $payloadFormat = WingApiRequest::PAYLOAD_FORMAT;

    /* Class auth parameters */
    private $authToken = null;

    /* Class methods */
    /**
     * @method __construct
     *
     * Constructor
     *
     * @return void
     */
    public function __construct()
    {
    }

    /**
     * @method public setAuthToken
     *
     * @param string $authToken
     *
     * @return void
     */
    public function setAuthToken($authToken)
    {
        $this->authToken = $authToken;
    }

    /**
     * @method public setHTTPMethod
     *
     * @param string $method
     *
     * @return $this
     */
    public function setHTTPMethod($method)
    {
        $availableMethods = [
            WingApiRequest::GET,
            WingApiRequest::POST,
            WingApiRequest::DELETE,
            WingApiRequest::PUT,
        ];

        if (!in_array(Tools::strtoupper($method), $availableMethods)) {
            throw new Exception('[WingApiRequest][Code:' . __LINE__ . '] Given HTTPMethod (' . Tools::strtoupper($method) . ') is not available');
        }

        $this->HTTPMethod = Tools::strtoupper($method);

        return $this;
    }

    /**
     * @method public setBaseURL
     *
     * @param string $mode
     *
     * @return $this
     */
    public function setBaseURL($baseUrl)
    {
        $this->baseURL = $baseUrl;

        return $this;
    }

    /**
     * @method public setPayload
     *
     * @param array $payload
     *
     * @return $this
     */
    public function setPayload($payload)
    {
        $this->payload = $payload;

        return $this;
    }

    /**
     * @method public setPayloadFormat
     *
     * @param string $format
     *
     * @return $this
     */
    public function setPayloadFormat($format)
    {
        $availableFormats = [
            WingApiRequest::PAYLOAD_FORMAT,
            WingApiRequest::URLENCODED_FORMAT,
        ];

        if (!in_array(Tools::strtolower($format), $availableFormats)) {
            throw new Exception('[WingApiRequest][Code:' . __LINE__ . '] Given format (' . Tools::strtolower($format) . ') is not available');
        }

        $this->payloadFormat = Tools::strtolower($format);

        return $this;
    }

    /**
     * @method public executeRequest
     *
     * @return array
     */
    public function executeRequest()
    {
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => $this->buildURL(),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => false,
            true,
            CURLOPT_VERBOSE => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CUSTOMREQUEST => $this->HTTPMethod,
            CURLOPT_POSTFIELDS => $this->formatPayload(),
            CURLOPT_HTTPHEADER => $this->buildHeaders(),
            CURLOPT_FOLLOWLOCATION => true,
        ]);

        $response = curl_exec($curl);
        $err = curl_error($curl);
        curl_close($curl);

        if ($err) {
            return [
                'fail' => $err,
            ];
        }

        return [
            'success' => $response,
        ];
    }

    /**
     * @method private buildHeaders
     *
     * @return array
     */
    private function buildHeaders()
    {
        $mandatoryHeaders = [
            'Cache-control: no-cache',
            'Accept: application/json',
            'Content-type: application/json',
        ];

        if ($this->authToken) {
            $mandatoryHeaders[] = 'Authorization: Basic ' . $this->authToken;
        }

        return $mandatoryHeaders;
    }

    /**
     * @method private buildURL
     *
     * @return string
     */
    private function buildURL()
    {
        return $this->baseURL;
    }

    /**
     * @method private formatPayload
     *
     * @return string
     */
    private function formatPayload()
    {
        if (WingApiRequest::PAYLOAD_FORMAT == $this->payloadFormat) {
            return json_encode($this->payload);
        } elseif (WingApiRequest::URLENCODED_FORMAT == $this->payloadFormat) {
            return http_build_query($this->payload);
        } else {
            return '';
        }
    }
}
