<?php
/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License version 3.0
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/AFL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License version 3.0
 */
require_once _PS_MODULE_DIR_ .
    DIRECTORY_SEPARATOR .
    'wing' .
    DIRECTORY_SEPARATOR .
    'classes' .
    DIRECTORY_SEPARATOR .
    'helpers.php';

/**
 * @method public void run()
 */
class WingUpdateOrderStatusModuleFrontController extends ModuleFrontController
{
    private $shopId = null;

    public function initContent()
    {
        $this->checkCredentials();
        $pOrderRef = Tools::getValue('ref');
        $parameterOrderRef = !empty($pOrderRef) ? $pOrderRef : null;

        $preparedStatus = (int) Configuration::get(
            'WING_SHOP_' . $this->shopId . '_PREPARED_STATUS'
        );
        $shippedStatus = (int) Configuration::get(
            'WING_SHOP_' . $this->shopId . '_SHIPPED_STATUS'
        );
        $deliveredStatus = (int) Configuration::get(
            'WING_SHOP_' . $this->shopId . '_DELIVERED_STATUS'
        );

        if (!$parameterOrderRef) {
            $this->displayError('400', [
                'status' => 'bad_request',
                'message' => 'missing_ref',
            ]);
        }

        $order = new Order($parameterOrderRef);
        if (!Validate::isLoadedObject($order)) {
            WingHelpers::log('404 - Order not found: ' . $parameterOrderRef);
            $this->displayError('404', [
                'status' => 'order_not_found',
                'message' => 'order ' . $parameterOrderRef,
            ]);
        }

        /* match hook type */
        $event = Tools::getValue('event');

        if ('parcel:prepared' === $event) {
            $alreadyShipped = $order->getHistory(
                $order->id_lang,
                $shippedStatus,
                false
            );
    
            if ($alreadyShipped) {
                $prev = (object) $alreadyShipped[0];
                $response = [
                    'status' => 'alreadyFulfilled',
                    'fulfilledDate' => $prev->date_add,
                    'id_order_state' => $prev->id_order_state,
                    'state_name' => $prev->ostate_name,
                ];
                $this->displayResponse($response);
                exit;
            }

            $newStatus = $preparedStatus;
            WingHelpers::log('wingPreparedStatus:' . $newStatus);

            if (9999 === $newStatus) {
                $this->displayResponse(['status' => 'success']);
                exit;
            }
        }

        if ('parcel:delivered' === $event) {
            $newStatus = $deliveredStatus;
            WingHelpers::log('wingDeliveredStatus:' . $newStatus);

            if (9999 === $newStatus) {
                $this->displayResponse(['status' => 'success']);
                exit;
            }
        }

        if (!$newStatus) {
            $this->displayError('400', [
                'status' => 'bad_request',
                'message' => 'no status set',
            ]);
        }

        $history = new OrderHistory();
        $history->id_order = (int) $order->id;
        $history->date_add = date('Y-m-d H:i:s');
        $history->changeIdOrderState($newStatus, $order, true);
        $history->add();
        $this->displayResponse(['status' => 'success']);
    }

    private function checkCredentials()
    {
        $context = Context::getContext();
        $this->shopId = (int) $context->shop->id;
        $apiKey = Configuration::get('WING_SHOP_' . $this->shopId . '_API_KEY');
        $apiSecret = Configuration::get(
            'WING_SHOP_' . $this->shopId . '_API_SECRET'
        );
        $psSecurityToken = sha1($apiKey . ':' . $apiSecret);

        /* In case of previous Apache version or nginx server use */
        $requestHeadersToLower = !function_exists('getallheaders')
            ? WingHelpers::getallheaders()
            : getallheaders();

        $requestHeaders = array_change_key_case(
            $requestHeadersToLower,
            CASE_LOWER
        );
        if (false == array_key_exists('x-wing-token', $requestHeaders)) {
            $this->displayError('401', [
                'status' => 'unauthorized',
                'message' => 'missing_token',
            ]);
        } elseif ($requestHeaders['x-wing-token'] != $psSecurityToken) {
            $this->displayError('401', [
                'status' => 'unauthorized',
                'message' => 'invalid_token',
            ]);
        }
    }

    private function displayError($status, $error)
    {
        header('HTTP/1.1 ' . $status . ' Unauthorized');
        header('Content-Type:application/json;charset=utf-8', false);
        $json = json_encode($error);
        echo $json;
        exit;
    }

    private function displayResponse($response)
    {
        header('Content-Type:application/json;charset=utf-8');
        $json = json_encode($response);
        echo $json;
        exit;
    }
}
