<?php
/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License version 3.0
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/AFL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License version 3.0
 */
require_once _PS_MODULE_DIR_ .
    DIRECTORY_SEPARATOR .
    'wing' .
    DIRECTORY_SEPARATOR .
    'classes' .
    DIRECTORY_SEPARATOR .
    'helpers.php';

/**
 * @method public void run()
 */
class WingUpdateOrderInfoModuleFrontController extends ModuleFrontController
{
    private $shopId = null;

    public function initContent()
    {
        $this->checkCredentials();

        $pOrderRef = Tools::getValue('ref');
        $parameterOrderRef = !empty($pOrderRef) ? $pOrderRef : null;

        $pTrackingId = Tools::getValue('tracking');
        $parameterTrackingId = !empty($pTrackingId) ? $pTrackingId : null;

        if (!$parameterOrderRef) {
            $this->displayError('400', [
                'status' => 'bad_request',
                'message' => 'missing_ref',
            ]);
        }

        if (!$parameterTrackingId) {
            $this->displayError('400', [
                'status' => 'bad_request',
                'message' => 'missing_tracking',
            ]);
        }

        $order = new Order($parameterOrderRef);
        if (!Validate::isLoadedObject($order)) {
            WingHelpers::log('404 - Order not found: ' . $parameterOrderRef);
            $this->displayError('404', [
                'status' => 'order_not_found',
                'message' => 'order ' . $parameterOrderRef,
            ]);
        }

        $wingShippedStatus = (int) Configuration::get(
            'WING_SHOP_' . $this->shopId . '_SHIPPED_STATUS'
        );
        if (!$wingShippedStatus) {
            $wingShippedStatus = 4;
        }
        WingHelpers::log('wingShippedStatus:' . $wingShippedStatus);

        $previousFulfillment = $order->getHistory(
            $order->id_lang,
            $wingShippedStatus,
            false
        );

        if ($previousFulfillment) {
            $prev = (object) $previousFulfillment[0];
            $response = [
                'status' => 'alreadyFulfilled',
                'fulfilledDate' => $prev->date_add,
                'id_order_state' => $prev->id_order_state,
                'state_name' => $prev->ostate_name,
            ];
            $this->displayResponse($response);
            return;
        }

        $preparedStatus = (int) Configuration::get(
            'WING_SHOP_' . $this->shopId . '_PREPARED_STATUS'
        );
        $previousPrepared = $order->getHistory(
            $order->id_lang,
            $preparedStatus,
            false
        );

        if (!$previousPrepared) {
            $preparedHistory = new OrderHistory();
            $preparedHistory->id_order = (int) $order->id;
            $preparedHistory->date_add = date('Y-m-d H:i:s');
            $preparedHistory->changeIdOrderState($preparedStatus, $order, true);
            $preparedHistory->add();
        }
        
        /* Change order status, add a new entry in order history and send an e-mail to the customer if needed */
        $order->shipping_number = $parameterTrackingId;
        $order->update();

        $order_carrier = new OrderCarrier($order->getIdOrderCarrier());
        $order_carrier->tracking_number = $parameterTrackingId;
        $order_carrier->update();

        $carrier = new Carrier($order->id_carrier);
        $urlTracking = str_replace(
            '@',
            $parameterTrackingId,
            $carrier->url
        );
        $templateVarsHistoryUpdate = ['{followup}' => $urlTracking];
        $history = new OrderHistory();
        $history->id_order = (int) $order->id;
        $history->date_add = date('Y-m-d H:i:s');
        $history->changeIdOrderState($wingShippedStatus, $order, true);
        $history->addWithemail(true, $templateVarsHistoryUpdate);

        // /* Send mail to final customer */
        // $customer = new Customer((int)$order->id_customer);
        // if (!Validate::isLoadedObject($customer)) {
        //     throw new PrestaShopException('Can\'t load Customer object');
        // }
        // if (!Validate::isLoadedObject($order_carrier)) {
        //     throw new PrestaShopException('Can\'t load Carrier object');
        // }
        // $url = property_exists('order_carrier', 'url') ? $order_carrier->url : '@';
        // $templateVarsShippingEmail = array(
        //     '{followup}' => str_replace('@', $order->shipping_number, $url),
        //     '{firstname}' => $customer->firstname,
        //     '{lastname}' => $customer->lastname,
        //     '{id_order}' => $order->id,
        //     '{shipping_number}' => $order->shipping_number,
        //     '{order_name}' => $order->getUniqReference()
        // );

        // Mail::Send((int) $order->id_lang, 'in_transit', Mail::l('Package in transit', (int) $order->id_lang), $templateVarsShippingEmail,
        //     $customer->email, $customer->firstname . ' ' . $customer->lastname, null, null, null, null,
        //     _PS_MAIL_DIR_, true, (int) $order->id_shop);

        $this->displayResponse(['status' => 'success']);
    }

    private function checkCredentials()
    {
        $context = Context::getContext();
        $this->shopId = (int) $context->shop->id;
        $apiKey = Configuration::get('WING_SHOP_' . $this->shopId . '_API_KEY');
        $apiSecret = Configuration::get(
            'WING_SHOP_' . $this->shopId . '_API_SECRET'
        );
        $psSecurityToken = sha1($apiKey . ':' . $apiSecret);

        /* In case of previous Apache version or nginx server use */
        $requestHeadersToLower = !function_exists('getallheaders')
            ? WingHelpers::getallheaders()
            : getallheaders();

        $requestHeaders = array_change_key_case(
            $requestHeadersToLower,
            CASE_LOWER
        );
        if (false == array_key_exists('x-wing-token', $requestHeaders)) {
            $this->displayError('401', [
                'status' => 'unauthorized',
                'message' => 'missing_token',
            ]);
        } elseif ($requestHeaders['x-wing-token'] != $psSecurityToken) {
            $this->displayError('401', [
                'status' => 'unauthorized',
                'message' => 'invalid_token',
            ]);
        }
    }

    private function displayError($status, $error)
    {
        header('HTTP/1.1 ' . $status . ' Unauthorized');
        header('Content-Type:application/json;charset=utf-8', false);
        $json = json_encode($error);
        echo $json;
        exit;
    }

    private function displayResponse($response)
    {
        header('Content-Type:application/json;charset=utf-8');
        $json = json_encode($response);
        echo $json;
        exit;
    }
}
