<?php
/**
 * 2007-2021 PrestaShop
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to http://www.prestashop.com for more information.
 *
 *  @author    PrestaShop SA <contact@prestashop.com>
 *  @copyright 2007-2021 PrestaShop SA
 *  @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
 *  International Registered Trademark & Property of PrestaShop SA
 */

if (!defined('_PS_VERSION_')) {
    exit();
}

require_once _PS_MODULE_DIR_ .
    DIRECTORY_SEPARATOR .
    'wing' .
    DIRECTORY_SEPARATOR .
    'classes' .
    DIRECTORY_SEPARATOR .
    'helpers.php';

$CLASSES_DEPENDENCIES = array(
    _PS_MODULE_DIR_ . // File
        DIRECTORY_SEPARATOR .
        'wing' .
        DIRECTORY_SEPARATOR .
        'classes' .
        DIRECTORY_SEPARATOR .
        'operations' .
        DIRECTORY_SEPARATOR .
        'configurations.php',
    _PS_MODULE_DIR_ . // File
        DIRECTORY_SEPARATOR .
        'wing' .
        DIRECTORY_SEPARATOR .
        'classes' .
        DIRECTORY_SEPARATOR .
        'operations' .
        DIRECTORY_SEPARATOR .
        'hooks.php',
    _PS_MODULE_DIR_ . // File
        DIRECTORY_SEPARATOR .
        'wing' .
        DIRECTORY_SEPARATOR .
        'classes' .
        DIRECTORY_SEPARATOR .
        'operations' .
        DIRECTORY_SEPARATOR .
        'tabs.php',
    _PS_MODULE_DIR_ . // File
        DIRECTORY_SEPARATOR .
        'wing' .
        DIRECTORY_SEPARATOR .
        'classes' .
        DIRECTORY_SEPARATOR .
        'apis' .
        DIRECTORY_SEPARATOR .
        'channel.php',
    _PS_MODULE_DIR_ . // File
        DIRECTORY_SEPARATOR .
        'wing' .
        DIRECTORY_SEPARATOR .
        'classes' .
        DIRECTORY_SEPARATOR .
        'apis' .
        DIRECTORY_SEPARATOR .
        'account.php',
    _PS_MODULE_DIR_ . // File
        DIRECTORY_SEPARATOR .
        'wing' .
        DIRECTORY_SEPARATOR .
        'classes' .
        DIRECTORY_SEPARATOR .
        'operations' .
        DIRECTORY_SEPARATOR .
        'orders.php',
    _PS_MODULE_DIR_ . // File
        DIRECTORY_SEPARATOR .
        'wing' .
        DIRECTORY_SEPARATOR .
        'classes' .
        DIRECTORY_SEPARATOR .
        'operations' .
        DIRECTORY_SEPARATOR .
        'products.php',
    _PS_MODULE_DIR_ . // File
        DIRECTORY_SEPARATOR .
        'wing' .
        DIRECTORY_SEPARATOR .
        'classes' .
        DIRECTORY_SEPARATOR .
        'operations' .
        DIRECTORY_SEPARATOR .
        'carriers.php'
);

WingHelpers::autoload($CLASSES_DEPENDENCIES);

/**
 *
 * @method public   void     __construct()
 *
 * @method private  void     setContext()
 *
 * @method public   boolean  install()
 * @method public   boolean  uninstall()
 *
 * @method public   void     getContent()
 * @method private  void     displayFilledConfigurationPage()
 *
 * @method public   void     hookActionCarrierUpdate(int $int)
 * @method public   void     hookBackOfficeHeader()
 * @method public   void     hookActionAdminControllerSetMedia()
 */

class Wing extends Module
{
    /* Class constants */
    const PS_16_MIN_VERSION = '1.6.0.1';
    const PS_WING_COMPATIBILITY_VERSION = '1.5.0.17';

    /* Class parameters */
    protected $context; // Prestashop context
    protected $errors = array(); // Errors array to store errors in module operations

    /* Class API Related parameters */
    private $post_uri = '';
    // private $isConnected    = false;    // is API Class Instance has been configured properly

    /* Class Configuration Related parameters */
    private $configurations = null; // Configurations Class Instance
    protected $wingTabs = null; // Tabs Class Instance
    public $ps_versions_compliancy = array(
        'min' => self::PS_WING_COMPATIBILITY_VERSION,
        'max' => '1.7'
    );

    public $wing;

    /* Class methods */
    /**
     * @method __construct
     *
     * Constructor
     *
     * @return void
     */
    public function __construct()
    {
        /* 1. Get the Prestashop context */
        $this->setContext();

        /* 2. Define module's information */
        $this->name = 'wing';
        $this->tab = 'shipping_logistics';
        $this->version = 'v6.2.3';
        $this->author = 'Wing';
        $this->need_instance = 0;
        $this->bootstrap = true;
        $this->module_key = ''; /* To add before production */

        /* 3. Instanciate module */
        parent::__construct();

        /* Define module's descriptions for post add into PrestaShop */
        $this->displayName = $this->l('Wing');
        $this->description = $this->l(
            'At Wing, we pick, pack and ship your products. Shipping, reinvented.'
        );
        $this->confirmUninstall = $this->l(
            'Are you sure you want to uninstall the Wing Module ?'
        );

        /* 4. Initiate module's setup */
        $this->configurations = new WingOperationsConfigurations();
        $this->hooks = new WingOperationsHooks();
        $this->wingTabs = new WingOperationsTabs();
        $this->baseUrl = 'https://api-graphql-prisma.wing.eu';

        /* Map Wing API classes to object */
        $this->wing = new stdClass();
        $this->orders = new WingOperationsOrders();
        $this->products = new WingOperationsProducts();
        $this->carriers = new WingOperationsCarriers();
    }

    /**
     * @method public uninstall
     *
     * @return Context
     */
    private function setContext()
    {
        $this->context = Context::getContext();
    }

    /**
     * @method install
     *
     * Module installation main function
     * This function is triggered when users use the module install button in the PrestaShop website
     * Modules & Services tab > Wing > Install
     *
     * @return boolean
     */
    public function install()
    {
        $shopIds = Shop::getShops(true, null, true);

        /* 5. Core checks */
        WingHelpers::log(
            'Wing module installation - Step 1/3 OK : Version is compatible (' .
                _PS_VERSION_ .
                ')'
        );

        /* The PrestaShop website's server should have the php curl extension installed to operate properly */
        if (!extension_loaded('curl')) {
            $this->_errors[] = $this->l('Wing requires curl php extension');
            WingHelpers::log(
                'Wing module installation - Step 2/3 KO : Wing requires curl php extension'
            );
            return false;
        }
        WingHelpers::log(
            'Wing module installation - Step 2/3 OK : PHP Curl extension is installed'
        );

        /* 6. Start installation */
        if (
            parent::install() &&
            /* 6.1. Remove previous module configurations */
            $this->configurations->removeConfigurations($shopIds) &&
            /* 6.2. Add module tab in Prestashop sidebar */
            $this->wingTabs->install($this->context, $this->name) &&
            /* 6.3. Register Prestashop Hooks */
            version_compare(_PS_VERSION_, self::PS_16_MIN_VERSION, '<')
                ? true
                : $this->registerHook('actionAdminControllerSetMedia') &&
                    $this->registerHook('backOfficeHeader') &&
                    $this->registerHook('actionOrderStatusPostUpdate') &&
                    $this->registerHook('actionCarrierUpdate')
            // && WingOperationsConfigurations::installDb() Not needed yet
        ) {
            $shopId = (int) $this->context->shop->id;
            Configuration::updateValue(
                'WING_SHOP_' . $shopId . '_API_BASEURL',
                'https://api-graphql-prisma.wing.eu'
            );

            /*Configuration::updateValue(
                'WING_SHOP_' . $shopId . '_API_KEY',
                'SET_API_KEY'
            );

            Configuration::updateValue(
                'WING_SHOP_' . $shopId . '_API_SECRET',
                'SET_SECRET_KEY'
            );*/

            /* 7.A. if success, continue */
            WingHelpers::log(
                'Wing module installation - Step 3/3 OK : Module installed'
            );
            return true;
        }

        /* 7.B. if not, breaks the installation process */
        WingHelpers::log(
            'Wing module installation - Step 3/3 KO : Module installation failed'
        );
        return false;
    }

    /**
     * @method public uninstall
     *
     * Module uninstallation main function
     * This function is triggered when users use the module uninstall button in the PrestaShop website
     * Modules & Services tab > Wing > Uninstall
     *
     * @return boolean
     */
    public function uninstall()
    {
        $shopIds = Shop::getShops(true, null, true);

        if (
            parent::uninstall() &&
            $this->wingTabs->uninstall('AdminWing') &&
            $this->configurations->removeConfigurations($shopIds)
            // && WingOperationsConfigurations::uninstallDb() Not needed yet
        ) {
            WingHelpers::log(
                'Wing module uninstallation - Step 1/1 OK : Module uninstalled'
            );
            return true;
        }

        WingHelpers::log(
            'Wing module uninstallation - Step 1/1 KO : Module uninstallation failed'
        );
        return false;
    }

    /**
     * @method public getContent
     *
     * Module function called after installation
     *
     * @return resource
     */
    public function getContent()
    {
        /* 8. Define default templates vars */
        $this->context->smarty->assign(array(
            'post_uri' => $this->post_uri,
            'errors' => $this->errors,
            'is15' => (bool) version_compare(_PS_VERSION_, '1.6', '<')
        ));

        /* 9. Load module backoffice's stylesheet */
        $this->context->controller->addCSS(
            _PS_MODULE_DIR_ .
                DIRECTORY_SEPARATOR .
                'wing' .
                DIRECTORY_SEPARATOR .
                'views' .
                DIRECTORY_SEPARATOR .
                'css' .
                DIRECTORY_SEPARATOR .
                'back.css'
        );

        /* Configuration Form POST Endpoint */
        if (Tools::isSubmit('processConfiguration')) {
            return $this->processConfigurationForm();
        }

        /* 10. Define default Templates Vars */
        return $this->displayFilledConfigurationPage();
    }

    /**
     * @method private displayFilledConfigurationPage
     *
     * This function is the main function to handle
     * and display the module configuration page
     *
     * @param string $bandMessages
     *
     * @return resource
     */
    private function displayFilledConfigurationPage($bandMessages = '')
    {
        $shopId = (int) $this->context->shop->id;
        $shop = Shop::getShop($shopId);
        $psName = $shop['name'];

        /* Specific way to set a channelId that already exists */
        $channelIdSetParam = Tools::getValue('channelIdSet');
        if (!empty($channelIdSetParam)) {
            Configuration::updateValue(
                'WING_SHOP_' . $shopId . '_API_CHANNEL_ID',
                $channelIdSetParam
            );
            Tools::redirectAdmin(
                $this->context->link->getAdminLink('AdminModules') .
                    '&configure=wing'
            );
        }

        $itemFields = array(
            "PRODUCT_NAME",
            "PRODUCT_EAN13",
            "PRODUCT_UPC",
            "PRODUCT_REFERENCE",
            "PRODUCT_SUPPLIER_REFERENCE",
            "EAN13",
            "UPC",
            "REFERENCE",
            "SUPPLIER_REFERENCE"
        );

        /* Get Prestashop statuses */
        $orderStatuses = OrderState::getOrderStates(
            $this->context->language->id
        );

        $orderStatuses[] = ['id_order_state' => 9999, 'name' => 'Aucun', 'color' => '#000'];

        $warehousesList = Warehouse::getWarehouses();

        /* Values to fill in the form */
        $preparedStatus = Configuration::get(
            'WING_SHOP_' . $shopId . '_PREPARED_STATUS'
        );

        $warehouses = Tools::jsonDecode(
            Configuration::get('WING_SHOP_' . $shopId . '_WAREHOUSES')
        );
        if (!$warehouses) {
            $warehouses = array();
        }

        $shippedStatus = Configuration::get(
            'WING_SHOP_' . $shopId . '_SHIPPED_STATUS'
        );
        $deliveredStatus = Configuration::get(
            'WING_SHOP_' . $shopId . '_DELIVERED_STATUS'
        );
        $syncStatuses = Tools::jsonDecode(
            Configuration::get('WING_SHOP_' . $shopId . '_SYNC_STATUSES')
        );
        if (!$syncStatuses) {
            $syncStatuses = array();
        }

        $apiBaseUrl = Configuration::get(
            'WING_SHOP_' . $shopId . '_API_BASEURL'
        );
        $designation =
            Configuration::get('WING_SHOP_' . $shopId . '_DESIGNATION_FIELD') !=
            null
                ? Configuration::get(
                    'WING_SHOP_' . $shopId . '_DESIGNATION_FIELD'
                )
                : $itemFields[0];
        $sku =
            Configuration::get('WING_SHOP_' . $shopId . '_SKU_FIELD') != null
                ? Configuration::get('WING_SHOP_' . $shopId . '_SKU_FIELD')
                : $itemFields[7];
        $ean =
            Configuration::get('WING_SHOP_' . $shopId . '_EAN_FIELD') != null
                ? Configuration::get('WING_SHOP_' . $shopId . '_EAN_FIELD')
                : $itemFields[5];

        $weight = Configuration::get('PS_WEIGHT_UNIT');

        $carriersConfig = Tools::jsonDecode(
            Configuration::get('WING_SHOP_' . $shopId . '_CARRIER_CONF')
        );

        /* Assign all variables to the view */
        $this->context->smarty->assign(array(
            'post_uri' =>
                $this->context->link->getAdminLink('AdminModules') .
                '&configure=' .
                $this->name .
                '&tab_module=' .
                $this->tab .
                '&module_name=' .
                $this->name .
                '&processConfiguration=1',
            'psName' => $psName,
            'wingModuleVersion' => $this->version,
            /* Form values */
            'orderStatuses' => $orderStatuses,
            'carriers' => $carriers,
            /* Saved values */
            'warehouses' => $warehouses,
            'warehousesList' => $warehousesList,
            'syncStatuses' => $syncStatuses,
            'preparedStatus' => $preparedStatus,
            'shippedStatus' => $shippedStatus,
            'deliveredStatus' => $deliveredStatus,
            'apiKey' => Configuration::get('WING_SHOP_' . $shopId . '_API_KEY'),
            'apiSecret' => Configuration::get(
                'WING_SHOP_' . $shopId . '_API_SECRET'
            ),
            'pickupName' => Configuration::get(
                'WING_SHOP_' . $shopId . '_PICKUP_NAME'
            ),
            'apiBaseUrl' => $apiBaseUrl,
            'designation' => $designation,
            'sku' => $sku,
            'ean' => $ean,
            'weight' => $weight,
            'itemFields' => $itemFields
        ));

        /* 11. Load and display configuration template */
        return $bandMessages .
            $this->context->smarty->fetch(
                _PS_MODULE_DIR_ .
                    DIRECTORY_SEPARATOR .
                    'wing' .
                    DIRECTORY_SEPARATOR .
                    'views' .
                    DIRECTORY_SEPARATOR .
                    'templates' .
                    DIRECTORY_SEPARATOR .
                    'admin' .
                    DIRECTORY_SEPARATOR .
                    'configure' .
                    DIRECTORY_SEPARATOR .
                    'configure.tpl'
            );
    }

    private function mapCarrierStatusToWingFormat($statusId)
    {
        /* Get Prestashop statuses */
        $orderStatuses = OrderState::getOrderStates(
            $this->context->language->id
        );
        /* Map entities to the final Order object */
        $obj = new stdClass();

        if (is_array($orderStatuses) || is_object($orderStatuses)) {
            foreach ($orderStatuses as $orderStatus) {
                if ($orderStatus['id_order_state'] == $statusId) {
                    $obj->psId = (int) $statusId;
                    $obj->name = $orderStatus['name'];

                    return $obj;
                }
            }
        }
    }

    private function processConfigurationForm()
    {
        /*** Step 1. Retrieve forms values ***/
        $formFields = array(
            'apiKey' => $this->l('API Key', 'wing'),
            'apiSecret' => $this->l('API Secret', 'wing'),
            'syncStatuses' => $this->l('Synchronization statuses', 'wing'),
        );

        $errors = array();
        foreach ($formFields as $key => $formField) {
            if (!Tools::getValue($key)) {
                $errors[] = $formField . ' ' . $this->l('field is missing');
            }
        }

        if (count($errors) > 0) {
            $errorMessage =
                'Wing module configuration - Step 1/4 (Retrieve Form Values): KO with following errors' .
                ' : ' .
                $this->l(
                    'The module configuration failed for the following reason',
                    'wing'
                ) .
                ' : <br /> - ' .
                join('<br /> - ', $errors);

            WingHelpers::log($errorMessage);

            return $this->displayFilledConfigurationPage(
                $this->displayError($errorMessage)
            );
        }

        WingHelpers::log(
            'Wing module configuration - Step 1/4 (Retrieve Form Values): OK'
        );

        /* Define context vars */
        $shopId = (int) $this->context->shop->id;
        $shop = Shop::getShop($shopId);
        $psIsMultiStore =
            sizeof(Shop::getCompleteListOfShopsID()) > 1 ? true : false;
        $psName = $shop['name'];
        $psUrl =
            strtok($this->context->link->getModuleLink($this->name), '?') .
            '?fc=module&module=wing';
        $psDomain = parse_url(
            $this->context->shop->getBaseURL(true),
            PHP_URL_HOST
        );
        $psUseHttps =
            Configuration::get('PS_SSL_ENABLED') || Tools::usingSecureMode()
                ? true
                : false;
        $psVersion = Configuration::get('PS_INSTALL_VERSION');

        /* Retrieve submitted values */
        /* Authentication form part */
        $apiKey = Tools::getValue('apiKey');
        $apiSecret = Tools::getValue('apiSecret');

        /* Statuses form part */
        $syncStatuses = Tools::getValue('syncStatuses');
        $syncStatusesWingFormat = array();
        if (is_array($syncStatuses) || is_object($syncStatuses)) {
            foreach ($syncStatuses as $syncStatus) {
                $syncStatusesWingFormat[] = $this->mapCarrierStatusToWingFormat(
                    $syncStatus
                );
            }
        }

        $preparedStatus = (int) Tools::getValue('preparedStatus');
        $preparedStatusWingFormat = $this->mapCarrierStatusToWingFormat(
            $preparedStatus
        );

        /* Warehouses form part */
        $warehouses = Tools::getValue('warehouses');
        $warehousesWingFormat = array();
        if (is_array($warehouses) || is_object($warehouses)) {
            foreach ($warehouses as $warehouse) {
                $warehousesWingFormat[] = (int) $warehouse;
            }
        }

        $shippedStatus = (int) Tools::getValue('shippedStatus');
        $shippedStatusWingFormat = $this->mapCarrierStatusToWingFormat(
            $shippedStatus
        );

        $deliveredStatus = (int) Tools::getValue('deliveredStatus');
        $deliveredStatusWingFormat = $this->mapCarrierStatusToWingFormat(
            $deliveredStatus
        );

        $apiBaseUrl = Tools::getValue('apiBaseUrl');
        $this->baseUrl = $apiBaseUrl;

        $designation = Tools::getValue('designationField');
        $sku = Tools::getValue('skuField');
        $ean = Tools::getValue('eanField');
        $pickupName = Tools::getValue('pickupName');
        $weight = strtoupper(Configuration::get('PS_WEIGHT_UNIT'));

        /* Generate Basic Auth token */
        $authToken = WingHelpers::base64UrlEncode($apiKey . ':' . $apiSecret);

        /* Define a Prestashop Security token */
        $psSecurityToken = sha1($apiKey . ':' . $apiSecret);

        /*** Step 2. Retrieve Wing Account ***/
        $this->wing->account = new WingApiAccount($authToken, $this->baseUrl);
        $getAccountResponse = $this->wing->account->get();

        $getAccountDecodedResponse = Tools::jsonDecode(
            $getAccountResponse['success']
        );

        if (
            isset($getAccountResponse['fail']) ||
            property_exists($getAccountDecodedResponse, 'errors')
        ) {
            $accountError = property_exists($getAccountResponse, 'fail')
                ? $getAccountResponse['fail']
                : $getAccountDecodedResponse->errors[0]->message;
            $getAccountErrorMessage =
                'Wing module configuration - Step 2/4 (Retrieve Wing Account): KO with the following error' .
                ' : ' .
                print_r($this->l($accountError, 'wing'), true);

            WingHelpers::log($getAccountErrorMessage);
            return $this->displayFilledConfigurationPage(
                $this->displayError($getAccountErrorMessage)
            );
        }

        $orgId = $getAccountDecodedResponse->data->whoami->org->id;

        WingHelpers::log(
            'Wing module configuration - Step 2/4 (Retrieve Wing Account): OK'
        );
        $channelId = Configuration::get(
            'WING_SHOP_' . $shopId . '_API_CHANNEL_ID'
        );

        /*** 3. Create Channel ***/
        $channel = [
            'name' => $psName,
            'organizationId' => $orgId,
            'type' => 'PRESTASHOP',
            'weightUnit' => $weight,
            'prestashop' => [
                'authToken' => $psSecurityToken,
                'name' => $psName,
                'shopId' => $shopId,
                'domain' => $psDomain,
                'useHttps' => $psUseHttps,
                'url' => $psUrl,
                'isMultiStore' => $psIsMultiStore,
                'version' => $psVersion ? $psVersion : '1.x',
                'moduleVersion' => $this->version,
                'syncStatuses' => $syncStatusesWingFormat,
                'shippedStatus' => $shippedStatusWingFormat,
                'designationField' => $designation,
                'skuField' => $sku,
                'eanField' => $ean
            ]
        ];

        /* Send channel's creation request */
        $this->wing->channel = new WingApiChannel($authToken, $this->baseUrl);

        if ($channelId) {
            $createChannelResponse = $this->wing->channel->updateChannel(
                $channelId,
                $channel
            );
        } else {
            $createChannelResponse = $this->wing->channel->createChannel(
                $channel
            );
        }

        if (isset($createChannelResponse['fail'])) {
            $errorMessage =
                'Wing module configuration - Step 3/4 (Create Wing Channel): KO with the following error' .
                ' : ' .
                print_r($this->l($createChannelResponse['fail'], 'wing'), true);
            WingHelpers::log(
                $errorMessage .
                    ' , ' .
                    'payload' .
                    ' : ' .
                    (version_compare(_PS_VERSION_, '1.6.0.1', '>=')
                        ? Tools::jsonEncode($createChannelResponse)
                        : $createChannelResponse)
            );
            return $this->displayFilledConfigurationPage(
                $this->displayError($errorMessage)
            );
        }

        $requestResponse = Tools::jsonDecode($createChannelResponse['success']);

        if ($requestResponse && property_exists($requestResponse, 'errors')) {
            $error = $requestResponse->errors[0]->message;

            $errorMessage =
                'Wing module configuration - Step 3/4 (Create Wing Channel): KO with the following error' .
                ' : ' .
                print_r($this->l($error, 'wing'), true);

            WingHelpers::log(
                $errorMessage .
                    ' , ' .
                    'payload' .
                    ' : ' .
                    (version_compare(_PS_VERSION_, '1.6.0.1', '>=')
                        ? Tools::jsonEncode($requestResponse)
                        : $requestResponse)
            );

            return $this->displayFilledConfigurationPage(
                $this->displayError($errorMessage)
            );
        }

        WingHelpers::log(
            'Wing module configuration - Step 3/4 (Create Wing Channel): OK' .
                ' : ' .
                'channel' .
                ' : ' .
                (version_compare(_PS_VERSION_, '1.6.0.1', '>=')
                    ? Tools::jsonEncode($channel)
                    : $channel)
        );

        /*** 4. Save Configuration ***/
        if (isset($requestResponse->data->createChannel)) {
            Configuration::updateValue(
                'WING_SHOP_' . $shopId . '_API_CHANNEL_ID',
                $requestResponse->data->createChannel->id
            );
        }

        Configuration::updateValue(
            'WING_SHOP_' . $shopId . '_API_BASEURL',
            $apiBaseUrl
        );

        Configuration::updateValue(
            'WING_SHOP_' . $shopId . '_API_KEY',
            $apiKey
        );
        Configuration::updateValue(
            'WING_SHOP_' . $shopId . '_API_SECRET',
            $apiSecret
        );
        Configuration::updateValue(
            'WING_SHOP_' . $shopId . '_PICKUP_NAME',
            $pickupName
        );
        Configuration::updateValue(
            'WING_SHOP_' . $shopId . '_BASIC_AUTH_TOKEN',
            $authToken
        );
        /* Save statuses configuration */
        Configuration::updateValue(
            'WING_SHOP_' . $shopId . '_SYNC_STATUSES',
            Tools::jsonEncode($syncStatuses)
        );
        Configuration::updateValue(
            'WING_SHOP_' . $shopId . '_PREPARED_STATUS',
            $preparedStatus
        );
        Configuration::updateValue(
            'WING_SHOP_' . $shopId . '_SHIPPED_STATUS',
            $shippedStatus
        );
        Configuration::updateValue(
            'WING_SHOP_' . $shopId . '_DELIVERED_STATUS',
            $deliveredStatus
        );
        /* Save synchronization configuration */
        Configuration::updateValue(
            'WING_SHOP_' . $shopId . '_WAREHOUSES',
            Tools::jsonEncode($warehouses)
        );
        Configuration::updateValue(
            'WING_SHOP_' . $shopId . '_DESIGNATION_FIELD',
            $designation
        );
        Configuration::updateValue('WING_SHOP_' . $shopId . '_SKU_FIELD', $sku);
        Configuration::updateValue('WING_SHOP_' . $shopId . '_EAN_FIELD', $ean);
        Configuration::updateValue(
            'WING_SHOP_' . $shopId . '_WEIGHT_FIELD',
            $weight
        );

        WingHelpers::log(
            'Wing module configuration - Step 4/4 (Save Configuration): OK'
        );

        return $this->displayFilledConfigurationPage(
            $this->displayConfirmation(
                $this->l('The module has been properly configured', 'wing')
            )
        );
    }

    /**
     * @method public hookBackOfficeHeader
     *
     * @return void
     */
    public function hookBackOfficeHeader()
    {
        if (version_compare(_PS_VERSION_, '1.6.0.1', '>=')) {
            $this->context->controller->addCSS(
                _PS_MODULE_DIR_ .
                    DIRECTORY_SEPARATOR .
                    'wing' .
                    DIRECTORY_SEPARATOR .
                    'views' .
                    DIRECTORY_SEPARATOR .
                    'css' .
                    DIRECTORY_SEPARATOR .
                    'tab.css'
            );
        }
    }
}
