<?php
/**
 * 2007-2021 PrestaShop
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to http://www.prestashop.com for more information.
 *
 *  @author    PrestaShop SA <contact@prestashop.com>
 *  @copyright 2007-2021 PrestaShop SA
 *  @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
 *  International Registered Trademark & Property of PrestaShop SA
 */

/**
 *
 * @method public $this    setAuthToken(string $authToken)
 * @method public $this    setHTTPMethod(string $method)
 * @method public $this    setBaseURL(string $mode)
 * @method public $this    setPayload(array $payload)
 * @method public $this    setPayloadFormat(string $format)
 * @method public array    executeRequest()
 * @method private array   buildHeaders()
 * @method private string  buildURL()
 * @method private string  formatPayload()
 */

if (!defined('_PS_VERSION_')) {
    exit();
}

require_once _PS_MODULE_DIR_ .
    DIRECTORY_SEPARATOR .
    'wing' .
    DIRECTORY_SEPARATOR .
    'classes' .
    DIRECTORY_SEPARATOR .
    'helpers.php';

class WingApiRequest
{
    /* Class constants */
    const POST = 'POST';
    const PUT = 'PUT';
    const DELETE = 'DELETE';
    const GET = 'GET';
    const URL_API_PRODUCTION = 'https://api-graphql-prisma.wing.eu';
    const URL_API_SANDBOX = '{{URL_API_SANDBOX}}';
    const URL_API_STAGING = 'https://api-graphql-prisma-staging.wing.eu';
    const URL_API_DEVELOPMENT = '{{URL_API_DEVELOPMENT}}';
    const PAYLOAD_FORMAT = 'json';
    const URLENCODED_FORMAT = 'urlencoded';

    /* Class parameters */
    private $baseURL = null;
    private $payload = array();
    private $HTTPMethod = WingApiRequest::GET;
    private $payloadFormat = WingApiRequest::PAYLOAD_FORMAT;

    /* Class auth parameters */
    private $authToken = null;

    /* Class methods */
    /**
     * @method __construct
     *
     * Constructor
     *
     * @return void
     */
    public function __construct()
    {
    }

    /**
     * @method public setAuthToken
     *
     * @param string $authToken
     *
     * @return void
     */
    public function setAuthToken($authToken)
    {
        $this->authToken = $authToken;
    }

    /**
     * @method public setHTTPMethod
     *
     * @param string $method
     *
     * @return $this
     */
    public function setHTTPMethod($method)
    {
        $availableMethods = array(
            WingApiRequest::GET,
            WingApiRequest::POST,
            WingApiRequest::DELETE,
            WingApiRequest::PUT
        );

        if (!in_array(Tools::strtoupper($method), $availableMethods)) {
            throw new Exception(
                "[WingApiRequest][Code:" .
                    __LINE__ .
                    "] Given HTTPMethod (" .
                    Tools::strtoupper($method) .
                    ") is not available"
            );
        }

        $this->HTTPMethod = Tools::strtoupper($method);
        return $this;
    }

    /**
     * @method public setBaseURL
     *
     * @param string $mode
     *
     * @return $this
     */
    public function setBaseURL($baseUrl)
    {
        $this->baseURL = $baseUrl;
        return $this;
    }

    /**
     * @method public setPayload
     *
     * @param array $payload
     *
     * @return $this
     */
    public function setPayload($payload)
    {
        $this->payload = $payload;
        return $this;
    }

    /**
     * @method public setPayloadFormat
     *
     * @param string $format
     *
     * @return $this
     */
    public function setPayloadFormat($format)
    {
        $availableFormats = array(
            WingApiRequest::PAYLOAD_FORMAT,
            WingApiRequest::URLENCODED_FORMAT
        );

        if (!in_array(Tools::strtolower($format), $availableFormats)) {
            throw new Exception(
                "[WingApiRequest][Code:" .
                    __LINE__ .
                    "] Given format (" .
                    Tools::strtolower($format) .
                    ") is not available"
            );
        }

        $this->payloadFormat = Tools::strtolower($format);

        return $this;
    }

    /**
     * @method public executeRequest
     *
     * @return array
     */
    public function executeRequest()
    {
        $curl = curl_init();
        curl_setopt_array($curl, array(
            CURLOPT_URL => $this->buildURL(),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER,
            true,
            CURLOPT_VERBOSE => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CUSTOMREQUEST => $this->HTTPMethod,
            CURLOPT_POSTFIELDS => $this->formatPayload(),
            CURLOPT_HTTPHEADER => $this->buildHeaders()
        ));

        $response = curl_exec($curl);
        $err = curl_error($curl);
        curl_close($curl);

        //@TODO Format request return properly
        if ($err) {
            return array(
                "fail" => $err
            );
        }

        return array(
            "success" => $response
        );
    }

    /**
     * @method private buildHeaders
     *
     * @return array
     */
    private function buildHeaders()
    {
        $mandatoryHeaders = array(
            "Cache-control: no-cache",
            "Accept: application/json",
            "Content-type: application/json"
        );

        if ($this->authToken) {
            $mandatoryHeaders[] = "Authorization: Basic " . $this->authToken;
        }

        return $mandatoryHeaders;
    }

    /**
     * @method private buildURL
     *
     * @return string
     */
    private function buildURL()
    {
        return $this->baseURL;
    }

    /**
     * @method private formatPayload
     *
     * @return string
     */
    private function formatPayload()
    {
        if ($this->payloadFormat == WingApiRequest::PAYLOAD_FORMAT) {
            return Tools::jsonEncode($this->payload);
        } elseif ($this->payloadFormat == WingApiRequest::URLENCODED_FORMAT) {
            return http_build_query($this->payload);
        } else {
            return "";
        }
    }
}
