<?php
/**
 * 2007-2021 PrestaShop
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to http://www.prestashop.com for more information.
 *
 *  @author    PrestaShop SA <contact@prestashop.com>
 *  @copyright 2007-2021 PrestaShop SA
 *  @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
 *  International Registered Trademark & Property of PrestaShop SA
 */

if (!defined("_PS_VERSION_")) {
    exit();
}

/**
 *
 * @method public   void        __construct()
 * @method public   array       getProducts(int $shopId, int $page, int $size, string $sku, string $barcode)
 */

class WingOperationsProducts
{
    protected $context;

    /* Class methods */
    /**
     * @method __construct
     *
     * Constructor
     *
     * @return void
     */
    public function __construct()
    {
        $this->setContext();
    }

    /**
     * @method public uninstall
     *
     * @return Context
     */
    private function setContext()
    {
        $this->context = Context::getContext();
    }

    /**
     * @method public getProducts
     *
     * @return array
     */
    public function getProducts($shopId, $page, $size, $sku, $barcode)
    {
        $ps_stock_available = _DB_PREFIX_ . "stock_available";
        $ps_product_attribute = _DB_PREFIX_ . "product_attribute";
        $ps_product = _DB_PREFIX_ . "product";
        $ps_product_lang = _DB_PREFIX_ . "product_lang";
        $db = Db::getInstance();

        $skuQuery = $sku
            ? "AND IF(ISNULL(pa.reference), p.reference, pa.reference) = '$sku'"
            : "";
        $barcodeQuery = $barcode
            ? "AND IF(ISNULL(pa.ean13), p.ean13, pa.ean13) = '$barcode'"
            : "";

        $offset = ($page - 1) * $size;

        $lang_default = (int) Configuration::get('PS_LANG_DEFAULT');

        $products = $db->executeS(
            "SELECT sa.id_product,
                    sa.id_product_attribute,
                    pl.name,
                    # variant sku exists only in `ps_product_attribute` table
                    IF(ISNULL(pa.reference), p.reference, pa.reference)   AS reference,
                    sa.quantity,
                    pl.description,
                    CAST(IF(pa.price > 0, pa.price, p.price) AS DECIMAL(10,2))    AS price,
                    CAST(IF(pa.weight > 0, pa.weight, p.weight) AS DECIMAL(10,2)) AS weight,
                    CAST(width AS DECIMAL(10,2))                                  AS width,
                    CAST(height AS DECIMAL(10,2))                                 AS height,
                    CAST(depth AS DECIMAL(10,2))                                  AS depth,
                    active,
                    IF(ISNULL(pa.ean13), p.ean13, pa.ean13)               as ean13
            FROM $ps_stock_available sa
                # use LEFT JOIN to include all records from left side
                # because `id_product_attribute = 0` don't exist on `ps_product_attribute` table
                LEFT JOIN $ps_product_attribute pa ON sa.id_product_attribute = pa.id_product_attribute
                JOIN $ps_product p ON sa.id_product = p.id_product
                LEFT JOIN $ps_product_lang pl
                    ON pl.id_lang = $lang_default
                    AND pl.id_product = sa.id_product
            WHERE sa.id_shop = $shopId
            AND (sa.id_product_attribute != 0
            OR (
                SELECT COUNT(*)
                FROM $ps_product_attribute t
                WHERE t.id_product = sa.id_product
                AND t.id_product_attribute != 0) = 0
                )
            $skuQuery $barcodeQuery
            ORDER BY sa.id_product ASC, pa.id_product_attribute ASC
            LIMIT $offset, $size"
        );

        if ($products === false) {
            return ["error" => $db->getMsgError()];
        }

        // same query, only `COUNT(*)` of total fields without limit
        $result = $db->getValue(
            "SELECT COUNT(*)
            FROM $ps_stock_available sa
                # use LEFT JOIN to include all records from left side
                # because `id_product_attribute = 0` don't exist on `ps_product_attribute` table
                LEFT JOIN $ps_product_attribute pa ON sa.id_product_attribute = pa.id_product_attribute
                JOIN $ps_product p ON sa.id_product = p.id_product
                LEFT JOIN $ps_product_lang pl ON sa.id_product = pl.id_product AND pl.id_lang = $lang_default
            WHERE sa.id_shop = $shopId
            AND (sa.id_product_attribute != 0
            OR (
                SELECT COUNT(*)
                FROM $ps_product_attribute t
                WHERE t.id_product = sa.id_product
                AND t.id_product_attribute != 0) = 0
            )
            $skuQuery $barcodeQuery
            ORDER BY sa.id_product ASC, pa.id_product_attribute ASC"
        );

        return [
            "shopId" => $shopId,
            "page" => $page,
            "size" => $size,
            "total" => (int) $result,
            "data" => $products
        ];
    }

    // we need sku to refetch product on success update
    public function updateProduct(
        $shopId,
        $sku,
        $id_product,
        $id_product_attribute,
        $quantity
    ) {
        $product = $this->getProducts($shopId, 1, 1, $sku, null)["data"][0];
        if (!$product) {
            return ["error" => "Product " . $sku . " not found"];
        }
        $total_qty = (int) $product["quantity"];
        $delta = $quantity - $total_qty;
        StockAvailable::updateQuantity(
            $id_product,
            $id_product_attribute,
            $delta,
            $shopId
        );
        $product["quantity"] = $total_qty + $delta;
        return $product;
    }
}
