<?php
/**
 * 2007-2021 PrestaShop
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to http://www.prestashop.com for more information.
 *
 *  @author    PrestaShop SA <contact@prestashop.com>
 *  @copyright 2007-2021 PrestaShop SA
 *  @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
 *  International Registered Trademark & Property of PrestaShop SA
 */

if (!defined('_PS_VERSION_')) {
    exit();
}

/**
 *
 * @method public   void        __construct()
 * @method public   array       getOrder(int $orderId)
 * @method public   array       getOrderIds(string $syncStatusesString, int $page, int $size, string $updatedAfter)
 * @method public   array       getOrders(string $syncStatusesString, int $page, int $size, string $createdAfter)
 * @method public   array       mapOrderProducts(array $orderProducts)
 * @method public   array       mapOrder(array $data)
 */

class WingOperationsOrders
{
    protected $context;

    /* Class methods */
    /**
     * @method __construct
     *
     * Constructor
     *
     * @return void
     */
    public function __construct()
    {
        $this->setContext();
    }

    /**
     * @method public uninstall
     *
     * @return Context
     */
    private function setContext()
    {
        $this->context = Context::getContext();
    }

    /**
     * @method public getOrder
     *
     * @param int     $orderId
     *
     * @return array
     */
    public function getOrder($orderId)
    {
        /* Get Order information */
        $order = new Order($orderId);

        /* Get Order Delivery Address */
        $deliveryAddressId = (int) $order->id_address_delivery;
        $deliveryAddress = new Address((int) $deliveryAddressId);

        /* Get Order Customer */
        $customer = new Customer((int) $order->id_customer);

        /* Get Carrier */
        $carrierId = $order->id_carrier;
        $carrier = new Carrier((int) $carrierId);

        /* Get Products from Order */
        $products = array();

        if (version_compare(_PS_VERSION_, '1.7.7', '>')) {
            $ps_order_detail = _DB_PREFIX_ . "order_detail";
            $db = Db::getInstance();
            $products = array();
            $products = $db->executeS("SELECT * FROM $ps_order_detail od WHERE od.id_order = $orderId");
        } else {
            $products = $order->getProducts();

            if (count($products) == 0) {
                $products = $order->getCartProducts();
            }
        }

        /* Map entities to the final Order object */
        $obj = new stdClass();
        $obj->order = $order;
        $obj->deliveryAddress = $deliveryAddress;
        $obj->customer = $customer;
        $obj->carrier = $carrier;
        $obj->products = $products;

        return $obj;
    }

    /**
     * @method public getOrderIds
     *
     * @param string  $syncStatutesString
     * @param string  $updatedAfter
     * @param int     $shopId
     *
     * @return array
     */
    public function getOrderIds(
        $syncStatusesString,
        $updatedAfter,
        $shopId,
        $page,
        $size
    ) {
        // Retrieve collection status
        $orderIdsArray = array();
        $syncStatuses = explode(',', $syncStatusesString);

        // Check if collections status has been set
        if (!$syncStatuses) {
            return;
        }

        $syncStatusString = $syncStatusesString
            ? ' AND `current_state` IN (' . $syncStatusesString . ')'
            : '';
        $updatedAfterString = $updatedAfter
            ? ' AND `date_upd` >= \'' . $updatedAfter . '\''
            : '';

        $ordersArray = Db::getInstance()->executeS(
            ' SELECT `id_order`,`id_cart`,`id_carrier`' .
                ' FROM `' .
                _DB_PREFIX_ .
                'orders`' .
                ' WHERE `id_shop` = (' .
                $shopId .
                ')' .
                $syncStatusString .
                $updatedAfterString .
                ' LIMIT ' .
                ($page - 1) * $size .
                ',' .
                $size
        );

        $totalOrderIds = count(
            Db::getInstance()->executeS(
                ' SELECT `id_order`,`id_cart`,`id_carrier`' .
                    ' FROM `' .
                    _DB_PREFIX_ .
                    'orders`' .
                    ' WHERE `id_shop` = (' .
                    $shopId .
                    ')' .
                    $syncStatusString .
                    $updatedAfterString
            )
        );

        $productWarehouses = array();
        $warehouses = Tools::jsonDecode(
            Configuration::get('WING_SHOP_' . $shopId . '_WAREHOUSES')
        );

        foreach ($ordersArray as $order) {
            if ($order != null) {
                $wingCarrierInfo = $this->mapOrderWingCarrier(
                    $order['id_carrier'],
                    (int) $order['id_cart']
                );
                if (
                    $wingCarrierInfo &&
                    $wingCarrierInfo->carrier != "NO_CARRIER_MATCHED"
                ) {
                    if ($warehouses && sizeof($warehouses) > 0) {
                        $productWarehouses = Db::getInstance()->executeS(
                            'SELECT od.product_id, wpl.id_warehouse' .
                            ' FROM `' .
                            _DB_PREFIX_ . 'order_detail` od,`' .
                            _DB_PREFIX_ . 'warehouse_product_location` wpl' .
                            ' WHERE od.id_order = (' . $order['id_order'] . ')' .
                            ' AND od.product_id = wpl.id_product'
                        );

                        $itemsWarehousesArr = array();

                        foreach ($productWarehouses as $item) {
                            $itemsWarehousesArr[$item['product_id']][] = $item['id_warehouse'];
                        }

                        $isAllowedWarehouses = true;
                        $healthcheck = false;
                        foreach($itemsWarehousesArr as $productWarehouses) {
                            $notFound = true;
                            foreach ($warehouses as $warehouse) {
                                $val = in_array($warehouse, $productWarehouses);

                                if ($val) {
                                    $notFound = false;
                                    break;
                                }
                            }

                            if ($notFound) {
                                $isAllowedWarehouses = false;
                                $healthcheck = true;
                                break;
                            }
                        }

                        if (!$isAllowedWarehouses && !$healthcheck) {
                            $orderWarehouses = Db::getInstance()->executeS(
                                'SELECT od.id_warehouse' .
                                ' FROM `' . _DB_PREFIX_ . 'order_detail` od' .
                                ' WHERE `id_order` = (' . $order['id_order'] . ')' .
                                ' GROUP BY od.id_warehouse'
                            );    

                            foreach ($orderWarehouses as $orderWarehouse) {
                                $val = in_array($orderWarehouse['id_warehouse'], $warehouses);

                                if ($val) {
                                    $isAllowedWarehouses = true;
                                }
                            }
                        }

                        if ($isAllowedWarehouses) {
                            $orderIdsArray[] = $order['id_order'];
                        }
                    } else {
                        $orderIdsArray[] = $order['id_order'];
                    }
                }
            }          
        }

        return array(
            "count" => $totalOrderIds,
            "orderIds" => $orderIdsArray
        );
    }

    /**
     * @method public getOrders
     *
     * @param string  $syncStatutesString
     * @param int     $page
     * @param int     $size
     * @param string  $createdAfter
     *
     * @return array
     */
    public function getOrders($syncStatusesString, $page, $size, $createdAfter)
    {
        $orders = array();
        $syncStatuses = explode(',', $syncStatusesString);

        if (!$syncStatuses) {
            $this->quickLog('No order status set');
            return;
        }

        $db = Db::getInstance();
        $orderIds = $db->executeS(
            ' SELECT `id_order`' .
                ' FROM `' .
                _DB_PREFIX_ .
                'orders`' .
                ' WHERE `current_state` IN (' .
                $syncStatusesString .
                ')' .
                ($createdAfter
                    ? ' AND `date_add` >= \'' . $createdAfter . '\''
                    : '') .
                ' LIMIT ' .
                ($page - 1) * $size .
                ',' .
                $size
        );

        if ($orderIds === false) {
            return ["error" => $db->getMsgError()];
        }

        foreach ($orderIds as $orderId) {
            $orderData = $this->getOrder($orderId['id_order']);
            $order = $this->mapOrder($orderData);

            if ($order != null) {
                $orders[] = $order;
            }
        }

        return array(
            "count" => count($orders),
            "page" => $page,
            "size" => $size,
            "orders" => $orders
        );
    }

    /**
     * @method private mapOrderProducts
     *
     * @param array    $orderProducts
     *
     * @return array
     */
    private function mapOrderProducts($orderProducts)
    {
        $products = array();

        foreach ($orderProducts as $orderProductToMap) {
            $orderProduct = (object) $orderProductToMap;
            $products[] = array(
                "product_id" => (int) $orderProduct->product_id,
                "product_attribute_id" =>
                    (int) $orderProduct->product_attribute_id,
                "product_name" => (string) $orderProduct->product_name,
                "product_quantity" => (float) $orderProduct->product_quantity,
                "product_price" => (float) $orderProduct->product_price,
                "product_ean13" => (string) $orderProduct->product_ean13,
                "product_upc" => (string) $orderProduct->product_upc,
                "product_reference" =>
                    (string) $orderProduct->product_reference,
                "product_supplier_reference" =>
                    (string) $orderProduct->product_supplier_reference,
                "product_weight" => (float) $orderProduct->product_weight,
                "total_price_tax_incl" =>
                    (float) $orderProduct->total_price_tax_incl,
                "total_price_tax_excl" =>
                    (float) $orderProduct->total_price_tax_excl,
                "unit_price_tax_incl" =>
                    (float) $orderProduct->unit_price_tax_incl,
                "unit_price_tax_excl" =>
                    (float) $orderProduct->unit_price_tax_excl,
                "id_product" => (int) $orderProduct->id_product,
                "id_supplier" => (int) $orderProduct->id_supplier,
                "id_manufacturer" => (int) $orderProduct->id_manufacturer,
                "ean13" => (string) $orderProduct->product_ean13,
                "upc" => (string) $orderProduct->upc,
                "quantity" => (float) $orderProduct->quantity,
                "minimal_quantity" => (int) $orderProduct->minimal_quantity,
                "price" => (float) $orderProduct->price,
                "additional_shipping_cost" =>
                    (float) $orderProduct->additional_shipping_cost,
                "reference" => $orderProduct->reference,
                "supplier_reference" =>
                    (string) $orderProduct->supplier_reference,
                "width" => (float) $orderProduct->width,
                "height" => (float) $orderProduct->height,
                "depth" => (float) $orderProduct->depth,
                "weight" => (float) $orderProduct->weight
            );
        }

        return $products;
    }

    /**
     * @method private mapOrderWingCarrier
     *
     * @param int    $carrierId
     *
     * @return string
     */
    private function mapOrderWingCarrier($carrierId, $orderCartId)
    {
        /* Define context vars */
        $shopId = (int) $this->context->shop->id;
        $psCarrier = new Carrier((int) $carrierId);

        /* Default Wing carrier settings */
        $matchedCarrier = '';
        $signature = false;
        $pickupId = null;
        $email = null;

        if (in_array($psCarrier->external_module_name, ['colissimo', 'soflexibilite', 'socolissimo'])) {
            $pickupId = Db::getInstance()->getValue(
                'SELECT pickup.colissimo_id FROM ' .
                    _DB_PREFIX_ .
                    'colissimo_pickup_point pickup LEFT JOIN ' .
                    _DB_PREFIX_ .
                    'colissimo_cart_pickup_point cartpickup ON ' .
                    ' (cartpickup.id_colissimo_pickup_point = pickup.id_colissimo_pickup_point)' .
                    ' WHERE cartpickup.id_cart = ' .
                    (int) $orderCartId
            );

            /* fallback * */
            if (null == $pickupId) {
                $pickupId = Db::getInstance()->getValue(
                    'SELECT `prid` FROM `' .
                        _DB_PREFIX_ .
                        'colissimo_delivery_info` WHERE `id_cart` = ' .
                        (int) $orderCartId
                );
                $email = Db::getInstance()->getValue(
                    'SELECT `ceemail` FROM `' .
                        _DB_PREFIX_ .
                        'colissimo_delivery_info` WHERE `id_cart` = ' .
                        (int) $orderCartId
                );
            }

        }

        if ('chronopost' == $psCarrier->external_module_name && Module::isInstalled('chronopost')) {
            $pickupId = Db::getInstance()->getValue(
                'SELECT `id_pr` FROM `' .
                    _DB_PREFIX_ .
                    'chrono_cart_relais` WHERE `id_cart` = ' .
                    (int) $orderCartId
            );
        }

        if ('mondialrelay' == $psCarrier->external_module_name && Module::isInstalled('mondialrelay')) {
            $pickupId = Db::getInstance()->getValue(
                'SELECT `selected_relay_num` FROM `' .
                    _DB_PREFIX_ .
                    'mondialrelay_selected_relay` WHERE `id_cart` = ' .
                    (int) $orderCartId
            );
        }

        /* Map entities to the final Order object */
        $obj = new stdClass();
        $obj->carrier = $psCarrier->name;
        $obj->signature = false;
        $obj->pickupId = $pickupId;
        $obj->email = $email;

        return $obj;
    }

    /**
     * @method private mapOrder
     *
     * @param array    $data
     *
     * @return array
     */
    public function mapOrder($data, $shopId)
    {
        $wingCarrierInfo = $this->mapOrderWingCarrier(
            $data->order->id_carrier,
            (int) $data->order->id_cart
        );

        $wingPickupName = Configuration::get(
            'WING_SHOP_' . $shopId . '_PICKUP_NAME'
        );

        return array(
            "reference" => $data->order->reference,
            "id_order" => (string) $data->order->id,
            "id_shop_group" => (int) $data->order->id_shop_group,
            "id_shop" => (int) $data->order->id_shop,
            "current_state" => (int) $data->order->current_state,
            "carrier" => $wingCarrierInfo->carrier,
            "pickupId" => $wingCarrierInfo->pickupId,
            "autoCreateCollectFromPickupName" => $wingPickupName,
            "gift" => (bool) $data->order->gift,
            "gift_message" => $data->order->gift_message,
            "total_paid" => (float) $data->order->total_paid,
            "total_paid_tax_incl" => (float) $data->order->total_paid_tax_incl,
            "total_paid_tax_excl" => (float) $data->order->total_paid_tax_excl,
            "total_shipping" => (float) $data->order->total_shipping,
            "total_shipping_tax_incl" =>
                (float) $data->order->total_shipping_tax_incl,
            "total_shipping_tax_excl" =>
                (float) $data->order->total_shipping_tax_excl,
            "date_add" => $data->order->date_add,
            "currency" => "EUR",
            "products" => $this->mapOrderProducts($data->products),
            "customer" => array(
                "country" => Tools::strtoupper(
                    Country::getIsoById($data->deliveryAddress->id_country)
                ),
                "company" => $data->deliveryAddress->company,
                "lastname" => $data->deliveryAddress->lastname,
                "firstname" => $data->deliveryAddress->firstname,
                "address1" => $data->deliveryAddress->address1,
                "address2" => $data->deliveryAddress->address2,
                "postcode" => $data->deliveryAddress->postcode,
                "email" => $wingCarrierInfo->email
                    ? $wingCarrierInfo->email
                    : $data->customer->email,
                "city" => $data->deliveryAddress->city,
                "other" => $data->deliveryAddress->other,
                "phone" => $data->deliveryAddress->phone,
                "phone_mobile" => $data->deliveryAddress->phone_mobile,
                "vat_number" => $data->deliveryAddress->vat_number
            )
        );
    }
}
